import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, CheckCircle, Briefcase, FileText, Users, Target, TrendingUp, Award, User, Clock, Star, CheckCircle2, Sparkles, Heart, ArrowRight, Zap } from 'lucide-react'
import { motion } from 'framer-motion'
import './BookConsultation.css'

const BookConsultation = () => {
  const navigate = useNavigate()
  const [selectedGoal, setSelectedGoal] = useState('')
  const [consultationDate, setConsultationDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const goals = [
    { 
      id: 'resume', 
      name: 'Резюме', 
      icon: FileText,
      color: '#FF6B6B',
      description: 'Создание и оптимизация резюме',
      duration: '60 минут',
      result: 'Готовое резюме',
      steps: ['Анализ', 'Структура', 'Оптимизация']
    },
    { 
      id: 'interview', 
      name: 'Собеседование', 
      icon: Users,
      color: '#FFA07A',
      description: 'Подготовка к собеседованию',
      duration: '90 минут',
      result: 'Готовность к интервью',
      steps: ['Подготовка', 'Практика', 'Стратегия']
    },
    { 
      id: 'career-change', 
      name: 'Смена профессии', 
      icon: Target,
      color: '#FFB6C1',
      description: 'План перехода в новую сферу',
      duration: '120 минут',
      result: 'План действий',
      steps: ['Анализ навыков', 'Обучение', 'Переход']
    },
    { 
      id: 'coaching', 
      name: 'Коучинг', 
      icon: TrendingUp,
      color: '#FF8787',
      description: 'Долгосрочное карьерное развитие',
      duration: '90 минут',
      result: 'Стратегия развития',
      steps: ['Цели', 'Навыки', 'Продвижение']
    }
  ]

  const consultants = [
    { name: 'Елена Смирнова', experience: '12 лет', rating: 5, specialty: 'Резюме, Собеседование', clients: '2500+', success: '98%' },
    { name: 'Дмитрий Петров', experience: '15 лет', rating: 5, specialty: 'Собеседование, Коучинг', clients: '3200+', success: '96%' },
    { name: 'Анна Козлова', experience: '10 лет', rating: 5, specialty: 'Смена профессии, Коучинг', clients: '1800+', success: '94%' },
    { name: 'Михаил Волков', experience: '8 лет', rating: 5, specialty: 'Резюме, Карьерный рост', clients: '2100+', success: '97%' }
  ]

  const guarantees = [
    { icon: Award, title: 'Сертифицированные специалисты', text: 'Все консультанты имеют международные сертификаты', metric: '100%' },
    { icon: Clock, title: 'Гибкий график', text: 'Запишитесь на удобное для вас время', metric: '24/7' },
    { icon: CheckCircle2, title: 'Гарантия результата', text: '100% клиентов достигают поставленных целей', metric: '100%' },
    { icon: Heart, title: 'Индивидуальный подход', text: 'Персональное внимание к каждому клиенту', metric: '100%' }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedGoal && consultationDate) {
      const savedSessions = localStorage.getItem('careerSessions')
      const sessions = savedSessions ? JSON.parse(savedSessions) : []
      
      const goalData = goals.find(g => g.id === selectedGoal)
      const randomConsultant = consultants[Math.floor(Math.random() * consultants.length)]
      
      const newSession = {
        id: Date.now(),
        consultant: randomConsultant.name,
        topic: goalData.name,
        result: `Консультация запланирована. Ожидаемый результат: ${goalData.result}`,
        review: '',
        rating: 0,
        date: consultationDate
      }

      sessions.push(newSession)
      localStorage.setItem('careerSessions', JSON.stringify(sessions))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedGoal('')
        setConsultationDate('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="book-consultation">
      <div className="container">
        <motion.section 
          className="book-hero"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <h1 className="book-title">Назначить консультацию</h1>
          <p className="book-subtitle">
            Выберите цель и начните свой путь к успеху
          </p>
        </motion.section>

        <motion.section 
          className="consultation-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.2 }}
        >
          <div className="consultation-form-wrapper">
            <form className="consultation-form" onSubmit={handleSubmit}>
              <div className="form-group">
                <label className="form-label">
                  <Target size={28} strokeWidth={2} className="label-icon" />
                  Выберите цель консультации
                </label>
                <div className="goals-minimal-list">
                  {goals.map((goal, index) => {
                    const IconComponent = goal.icon
                    return (
                      <motion.button
                        key={goal.id}
                        type="button"
                        className={`goal-minimal-item ${selectedGoal === goal.id ? 'selected' : ''}`}
                        onClick={() => setSelectedGoal(goal.id)}
                        initial={{ opacity: 0, x: -20 }}
                        animate={{ opacity: 1, x: 0 }}
                        transition={{ delay: 0.1 + index * 0.1, duration: 0.4 }}
                        whileHover={{ x: 4, scale: 1.01, transition: { duration: 0.2 } }}
                        style={{ 
                          borderLeftColor: selectedGoal === goal.id ? goal.color : 'rgba(255, 107, 107, 0.2)',
                        }}
                      >
                        <div className="goal-minimal-icon" style={{ background: selectedGoal === goal.id ? `${goal.color}20` : 'rgba(255, 107, 107, 0.1)', borderColor: selectedGoal === goal.id ? goal.color : 'rgba(255, 107, 107, 0.2)' }}>
                          <IconComponent size={32} strokeWidth={2} color={selectedGoal === goal.id ? goal.color : '#FF6B6B'} />
                        </div>
                        <div className="goal-minimal-content">
                          <h3 className="goal-minimal-name">{goal.name}</h3>
                          <p className="goal-minimal-description">{goal.description}</p>
                          <div className="goal-minimal-steps">
                            {goal.steps.map((step, stepIndex) => (
                              <span key={stepIndex} className="goal-step-badge">
                                <Zap size={12} strokeWidth={2} />
                                {step}
                              </span>
                            ))}
                          </div>
                          <div className="goal-minimal-meta">
                            <span className="goal-meta-item">
                              <Clock size={16} strokeWidth={1.5} />
                              {goal.duration}
                            </span>
                            <span className="goal-meta-item">
                              <CheckCircle2 size={16} strokeWidth={1.5} />
                              {goal.result}
                            </span>
                          </div>
                        </div>
                        {selectedGoal === goal.id && (
                          <motion.div 
                            className="goal-check-minimal"
                            initial={{ scale: 0 }}
                            animate={{ scale: 1 }}
                            transition={{ type: "spring", stiffness: 200, damping: 15 }}
                          >
                            <CheckCircle size={24} strokeWidth={2} color={goal.color} />
                          </motion.div>
                        )}
                      </motion.button>
                    )
                  })}
                </div>
              </div>

              <div className="form-group date-group-minimal">
                <label htmlFor="consultationDate" className="date-label-minimal">
                  <Calendar size={28} strokeWidth={2} className="label-icon" />
                  <span className="date-label-text">Выберите дату консультации</span>
                </label>
                <input
                  type="date"
                  id="consultationDate"
                  className="form-input"
                  value={consultationDate}
                  onChange={(e) => setConsultationDate(e.target.value)}
                  min={minDate}
                  max={maxDate}
                  required
                />
              </div>

              <motion.button 
                type="submit" 
                className="submit-button" 
                disabled={!selectedGoal || !consultationDate}
                whileHover={{ scale: 1.02, x: 2 }}
                whileTap={{ scale: 0.98 }}
              >
                <span>Записаться на консультацию</span>
                <ArrowRight size={24} strokeWidth={2} className="btn-icon" />
              </motion.button>

              {isSubmitted && (
                <motion.div 
                  className="success-message"
                  initial={{ opacity: 0, scale: 0.8 }}
                  animate={{ opacity: 1, scale: 1 }}
                  transition={{ duration: 0.4 }}
                >
                  <CheckCircle size={32} strokeWidth={2} className="success-icon" />
                  <span>Вы успешно записались на консультацию!</span>
                </motion.div>
              )}
            </form>
          </div>
        </motion.section>

        <motion.section 
          className="guarantees-minimal-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.3 }}
        >
          <div className="guarantees-minimal-header">
            <h2 className="section-title">Наши гарантии</h2>
          </div>
          <div className="guarantees-minimal-grid">
            {guarantees.map((guarantee, index) => {
              const IconComponent = guarantee.icon
              return (
                <motion.div
                  key={guarantee.title}
                  className="guarantee-minimal-card"
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: 0.4 + index * 0.1, duration: 0.4 }}
                  whileHover={{ x: 4, scale: 1.01, transition: { duration: 0.2 } }}
                >
                  <div className="guarantee-metric-minimal">
                    <span>{guarantee.metric}</span>
                  </div>
                  <div className="guarantee-icon-minimal">
                    <IconComponent size={28} strokeWidth={2} color="#FF6B6B" />
                  </div>
                  <div className="guarantee-content-minimal">
                    <h3 className="guarantee-title-minimal">{guarantee.title}</h3>
                    <p className="guarantee-text-minimal">{guarantee.text}</p>
                  </div>
                </motion.div>
              )
            })}
          </div>
        </motion.section>

        <motion.section 
          className="consultants-minimal-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.4 }}
        >
          <div className="consultants-minimal-header">
            <h2 className="section-title">Наши консультанты</h2>
          </div>
          <div className="consultants-minimal-grid">
            {consultants.map((consultant, index) => (
              <motion.div
                key={consultant.name}
                className="consultant-minimal-card"
                initial={{ opacity: 0, scale: 0.9 }}
                animate={{ opacity: 1, scale: 1 }}
                transition={{ delay: 0.5 + index * 0.1, duration: 0.4 }}
                whileHover={{ x: 4, scale: 1.01, transition: { duration: 0.2 } }}
              >
                <div className="consultant-avatar-minimal">
                  <User size={32} strokeWidth={2} color="#FF6B6B" />
                  <div className="consultant-success-minimal">
                    <span>{consultant.success}</span>
                  </div>
                </div>
                <div className="consultant-content-minimal">
                  <h3 className="consultant-name-minimal">{consultant.name}</h3>
                  <p className="consultant-experience-minimal">{consultant.experience} опыта</p>
                  <div className="consultant-rating-minimal">
                    {Array.from({ length: 5 }, (_, i) => (
                      <Star key={i} size={16} fill="#FF6B6B" stroke="#FF6B6B" strokeWidth={2} />
                    ))}
                  </div>
                  <p className="consultant-specialty-minimal">{consultant.specialty}</p>
                  <div className="consultant-stats-minimal">
                    <div className="consultant-stat-minimal">
                      <Users size={16} strokeWidth={1.5} color="#4ECDC4" />
                      <span>{consultant.clients} клиентов</span>
                    </div>
                  </div>
                </div>
              </motion.div>
            ))}
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default BookConsultation
